(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     24198,        779]
NotebookOptionsPosition[     20935,        668]
NotebookOutlinePosition[     21328,        685]
CellTagsIndexPosition[     21285,        682]
WindowFrame->Normal
ContainsDynamic->False*)

(* Beginning of Notebook Content *)
Notebook[{

Cell[CellGroupData[{
Cell[TextData[{
 ".NET/Link Example: BZip2 Compression\n\n",
 StyleBox["This example demonstrates calling an \"unmanaged\" Windows DLL \
(that is, an old C-style DLL, not a .NET assembly) from ",
  FontFamily->"Arial",
  FontSize->12,
  FontWeight->"Plain",
  FontVariations->{"CompatibilityType"->0}],
 StyleBox["Mathematica",
  FontFamily->"Arial",
  FontSize->12,
  FontWeight->"Plain",
  FontSlant->"Italic",
  FontVariations->{"CompatibilityType"->0}],
 StyleBox[" using ",
  FontFamily->"Arial",
  FontSize->12,
  FontWeight->"Plain",
  FontVariations->{"CompatibilityType"->0}],
 StyleBox["DefineDLLFunction", "Input",
  FontFamily->"Arial",
  FontSize->12,
  FontVariations->{"CompatibilityType"->0}],
 StyleBox[". The DLL we use is libbz2-1.0.0.dll, which performs compression \
and decompression using the bzip2 algorithm. The DLL is bundled with this \
example file, so you can run the program immediately. The code for the DLL is \
open source, and the home page, which includes documentation, can be found at \
",
  FontFamily->"Arial",
  FontSize->12,
  FontWeight->"Plain",
  FontVariations->{"CompatibilityType"->0}],
 StyleBox[ButtonBox["http://sources.redhat.com/bzip2/",
  BaseStyle->"Hyperlink",
  ButtonData:>{
    URL["http://sources.redhat.com/bzip2/"], None}],
  FontSize->12],
 StyleBox[". \n\nWe define two ",
  FontFamily->"Arial",
  FontSize->12,
  FontWeight->"Plain",
  FontVariations->{"CompatibilityType"->0}],
 StyleBox["Mathematica",
  FontFamily->"Arial",
  FontSize->12,
  FontWeight->"Plain",
  FontSlant->"Italic",
  FontVariations->{"CompatibilityType"->0}],
 StyleBox[" functions, ",
  FontFamily->"Arial",
  FontSize->12,
  FontWeight->"Plain",
  FontVariations->{"CompatibilityType"->0}],
 StyleBox["Compress", "Input",
  FontFamily->"Arial",
  FontSize->12,
  FontVariations->{"CompatibilityType"->0}],
 StyleBox[" and ",
  FontFamily->"Arial",
  FontSize->12,
  FontWeight->"Plain",
  FontVariations->{"CompatibilityType"->0}],
 StyleBox["Uncompress", "Input",
  FontFamily->"Arial",
  FontSize->12,
  FontVariations->{"CompatibilityType"->0}],
 StyleBox[", that write and read compressed data.\n\n",
  FontFamily->"Arial",
  FontSize->12,
  FontWeight->"Plain",
  FontVariations->{"CompatibilityType"->0}],
 StyleBox["To try this example, evaluate all the initialization cells (you \
can do this with the menu command Kernel/Evaluation/Evaluate Initialization). \
Then go to the Example section.",
  FontSize->12,
  FontWeight->"Plain",
  FontVariations->{"CompatibilityType"->0}]
}], "Title"],

Cell[CellGroupData[{

Cell["Code and Discussion", "Section"],

Cell["Needs[\"NETLink`\"];", "Input",
 InitializationCell->True],

Cell[TextData[{
 "If the DLL was placed in a ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " application's library directory (for example, ",
 StyleBox["<Mathematica \
dir>\\AddOns\\Applications\\SomeApp\\Libraries\\Windows",
  FontFamily->"Courier New"],
 "), or anywhere on the system's PATH, then we could specify just the name of \
the dll, ",
 StyleBox["libbz2-1.0.0.dll",
  FontFamily->"Courier New"],
 ", and it would automatically be found. Because this DLL is in a \
non-standard location, however, we must supply the full path, which we obtain \
by noting that the DLL is in the same directory as the current notebook:"
}], "Text"],

Cell["\<\
fullDllPath = ToFileName[First[\"FileName\" /. \
NotebookInformation[InputNotebook[]]], \"libbz2-1.0.0.dll\"];\
\>", "Input",
 InitializationCell->True],

Cell[TextData[{
 "Now we use ",
 StyleBox["DefineDLLFunction", "Input",
  FontWeight->"Bold"],
 " to create ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " functions whose action is to call into the desired functions from the DLL. \
We need functions for opening and closing files, reading and writing \
compressed data, and checking the error state."
}], "Text"],

Cell[CellGroupData[{

Cell["bzopen", "Subsection"],

Cell[TextData[{
 "Here is the C-language prototype for the ",
 StyleBox["BZ2_bzopen", "Output"],
 " function:"
}], "Text"],

Cell["BZFILE * BZ2_bzopen  ( const char *path, const char *mode );", "Program"],

Cell[TextData[{
 "Now define the ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " function that calls this DLL function. Note that pointers to arbitrary \
structures like ",
 StyleBox["BZFILE*", "Output"],
 " are mapped to ",
 StyleBox["IntPtr", "Output"],
 ", the .NET class that represents a generic pointer. ",
 StyleBox["DefineDLLFunction", "Input"],
 " will accept syntax for type names in C, VB, or C# notation. Typically you \
are looking at a C-language prototype for a DLL function, so that is the most \
natural form to use:"
}], "Text"],

Cell["\<\
bzopen = DefineDLLFunction[\"BZ2_bzopen\", fullDllPath, \"IntPtr\", {\"const \
char*\", \"const char*\"}];\
\>", "Input",
 InitializationCell->True],

Cell["\<\
There are many alternative ways to define this function. Here are some \
examples:\
\>", "Text"],

Cell["\<\
(* Using const is unnecessary: *)
DefineDLLFunction[\"BZ2_bzopen\", fullDllPath, \"IntPtr\", {\"char*\", \
\"char*\"}];
(* This uses C# syntax for string args: *)
DefineDLLFunction[\"BZ2_bzopen\", fullDllPath, \"IntPtr\", {\"string\", \
\"string\"}];
(* This uses VB syntax for string args: *)
DefineDLLFunction[\"BZ2_bzopen\", fullDllPath, \"IntPtr\", {\"ByVal path As \
String\", \"ByVal mode As String\"}];\
\>", "Input",
 Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{

Cell["bzwrite", "Subsection"],

Cell[TextData[{
 "Here is the C-language prototype for the ",
 StyleBox["BZ2_bzwrite", "Output"],
 " function:"
}], "Text"],

Cell["int BZ2_bzwrite ( BZFILE* b, void* buf, int len );", "Program"],

Cell[TextData[{
 "Once again we use the generic .NET ",
 StyleBox["IntPtr", "Output"],
 " type for the ",
 StyleBox["BZFILE*", "Output"],
 " argument. The ",
 StyleBox["void* buf", "Output"],
 " argument is the address of a block of memory containing the data to \
compress and write out to the file. You could think of it as a ",
 StyleBox["byte*", "Output"],
 " or, even better, as a ",
 StyleBox["byte[]", "Output"],
 ". .NET/Link lets you pass a list to a pointer or array argument, and we \
want to pass a list of bytes from ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " for this argument, so it is clearest to declare it as a ",
 StyleBox["byte[]", "Output"],
 ". Note that we could also use ",
 StyleBox["char[]", "Output"],
 " to mean the same thing. .NET/Link lets you use C or C# syntax for these \
types, but it gives preference to C syntax, so even though ",
 StyleBox["char", "Output"],
 " in C and C# are different sizes, .NET/Link assumes you mean the C ",
 StyleBox["char", "Output"],
 " type (a single byte)."
}], "Text"],

Cell["\<\
bzwrite = DefineDLLFunction[\"BZ2_bzwrite\", fullDllPath, \"int\", \
{\"IntPtr\", \"byte[]\", \"int\"}];\
\>", "Input",
 InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["bzread", "Subsection"],

Cell[TextData[{
 "Here is the C-language prototype for the ",
 StyleBox["BZ2_bzread", "Output"],
 " function:"
}], "Text"],

Cell["int BZ2_bzread  ( BZFILE* b, void* buf, int len );", "Program"],

Cell[TextData[{
 "Compare this to the definition for ",
 StyleBox["bzwrite", "Input"],
 ". The ",
 StyleBox["void* buf", "Output"],
 " argument represents the address of a block of memory into which data will \
be read. You can specify ",
 StyleBox["void*", "Output"],
 " as the type and it will be automatically mapped to ",
 StyleBox["IntPtr", "Output"],
 ", but for our uses we want it to be typed specifically as a ",
 StyleBox["byte*", "Output"],
 " or ",
 StyleBox["byte[]", "Output"],
 " (either works fine)."
}], "Text"],

Cell["\<\
bzread = DefineDLLFunction[\"BZ2_bzread\", fullDllPath, \"int\", {\"IntPtr\", \
\"byte[]\", \"int\"}];\
\>", "Input",
 InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["bzclose", "Subsection"],

Cell[TextData[{
 "Here is the C-language prototype for the ",
 StyleBox["BZ2_bzclose", "Output"],
 " function:"
}], "Text"],

Cell["void BZ2_bzclose ( BZFILE* b );", "Program"],

Cell["\<\
bzclose = DefineDLLFunction[\"BZ2_bzclose\", fullDllPath, \"void\", {\"IntPtr\
\"}];\
\>", "Input",
 InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["bzerror", "Subsection"],

Cell[TextData[{
 "Here is the C-language prototype for the ",
 StyleBox["BZ2_bzerror", "Output"],
 " function:"
}], "Text"],

Cell["char* BZ2_bzerror ( BZFILE* b, int* errCode );", "Program"],

Cell[TextData[{
 "The last argument is an ",
 StyleBox["int*", "Output"],
 " that gets filled in with an error code by the function. We could declare \
this as an ",
 StyleBox["int*", "Output"],
 ", but there is a better way, for the following reasons. .NET/Link lets you \
pass a literal integer, like 42, for an ",
 StyleBox["int*", "Output"],
 " parameter, but that is only suitable when the argument is \"in\" only, \
because there is no way of communicating back out any changes made to the \
integer. To solve this, you can pass a symbol like ",
 StyleBox["x", "Output"],
 " to an ",
 StyleBox["int*", "Output"],
 " slot, and ",
 StyleBox["x", "Output"],
 " will be assigned the value of the integer written into the ",
 StyleBox["int*", "Output"],
 " address by the function call. But .NET/Link cannot know that this is \
effectively an out-only parameter, so .NET/Link requires that a legitimate \
integer value be passed in. This means that even though the value of the \
integer is not inspected by the function, you need to initialize the value by \
setting ",
 StyleBox["x", "Output"],
 " to, say, 42 before passing ",
 StyleBox["x", "Output"],
 " to the function. A slicker trick is to tell .NET/Link that this is an \
out-only parameter by declaring it in C#-style syntax as ",
 StyleBox["out int", "Output"],
 ". Note that the pointer asterisk is gone from this--an ",
 StyleBox["out int", "Output"],
 ", like a ",
 StyleBox["ref int", "Output"],
 " or ",
 StyleBox["ByRef As Integer", "Output"],
 " (in VB syntax), is effectively already the address of an int. When we \
declare the parameter as ",
 StyleBox["out int", "Output"],
 ", .NET/Link knows to ignore its value going in to the function, but \
propagate a value back out. This means that we can paass in an uninitialized \
symbol like ",
 StyleBox["x", "Output"],
 " to the function and then inspect the value of ",
 StyleBox["x", "Output"],
 " after the call to see what integer value it was assigned during the call."
}], "Text"],

Cell["\<\
bzerror = DefineDLLFunction[\"BZ2_bzerror\", fullDllPath, \"char*\", \
{\"IntPtr\", \"out int\"}];\
\>", "Input",
 InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["The Compress function", "Subsection"],

Cell["\<\
Compress::fopen = \"Failed to open file `1` for writing.\";
Compress::bzerr = \"An error was reported by a bzip2 library routine. The \
error code was `1` and the message was `2`.\";\
\>", "Input",
 InitializationCell->True],

Cell[TextData[{
 "The main argument for ",
 StyleBox["Compress", "Input"],
 " is a ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " string, but we want to pass the data into the DLL as an array (List) of \
bytes, so we convert strings to byte arrays. ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " strings consist of 2-byte Unicode characters, so to properly handle \
non-ASCII characters like \[Alpha] we have to either convert every character \
to 2 bytes before compression (not a very good idea to double the size of the \
data before starting compression!) or encode in an efficient format like UTF8 \
that lets ASCII characters remain as single bytes."
}], "Text"],

Cell["\<\
Compress[file_String, data_String] := Compress[file, ToCharacterCode[data, \
\"UTF8\"]]

Compress[file_String, data:{___Integer}] :=
\tNETBlock[
\t\tModule[{bzPtr, numBytesWritten, errString, errCode},
\t\t\tbzPtr = bzopen[file, \"w\"];
\t\t\t(* bzPtr is a pointer (an instance of the IntPtr class). Here we test \
to see if it was NULL: *)
\t\t\tIf[bzPtr@ToInt32[] == 0,
\t\t\t\tMessage[Compress::fopen, file];
\t\t\t\tReturn[$Failed]
\t\t\t];
\t\t\tnumBytesWritten = bzwrite[bzPtr, data, Length[data]];
\t\t\tIf[numBytesWritten == 0,
\t\t\t\terrString = bzerror[bzPtr, errCode];
\t\t\t\tMessage[Compress::bzerr, errCode, errString];
\t\t\t\tnumBytesWritten = $Failed
\t\t\t];
\t\t\tbzclose[bzPtr];
\t\t\tnumBytesWritten
\t\t]
\t]\
\>", "Input",
 InitializationCell->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["The Uncompress function", "Subsection"],

Cell["\<\
Uncompress::fopen = \"Failed to open file `1` for reading.\";
Uncompress::bzerr = \"An error was reported by a bzip2 library routine. The \
error code was `1` and the message was `2`.\";\
\>", "Input",
 InitializationCell->True],

Cell[TextData[{
 "The ",
 StyleBox["Uncompress", "Input"],
 " function is slightly trickier. There are two ways to do it, and we will \
examine each in turn. The two techniques differ in how they handle the second \
argument to ",
 StyleBox["BZ2_bzread", "Output"],
 ", which is the address of a buffer into which the function will write \
uncompressed data.\n\nThe first technique is the simplest. It types the \
second argument to ",
 StyleBox["BZ2_bzread", "Output"],
 " as a ",
 StyleBox["byte[]", "Output"],
 " (this was done earlier in the definition for ",
 StyleBox["bzread", "Input"],
 ") and makes use of .NET's default marshaling for arays. When a managed \
array of bytes is sent to a DLL function, it is marshaled as a a native \
C-style array of bytes, which is just a contiguous block of memory, exactly \
what the ",
 StyleBox["BZ2_bzread", "Output"],
 " function wants. When the DLL call returns, .NET marshals the data from the \
block of memory back into the managed array. Therefore we can allocate an \
array of bytes in .NET, pass this object as the second argument to ",
 StyleBox["bzread", "Input"],
 ", and then when the function returns we can simply read the contents of \
that array to get the compressed data."
}], "Text"],

Cell["\<\
Uncompress[file_String] :=
\tNETBlock[
\t\tModule[{bzPtr, bufPtr, cnt, result, managedArray, errString, errCode},
\t\t\tbzPtr = bzopen[file, \"r\"];
\t\t\tIf[bzPtr@ToInt32[] == 0,
\t\t\t\tMessage[Uncompress::fopen, file];
\t\t\t\tReturn[$Failed]
\t\t\t];
\t\t\t(* Here we allocate a .NET array of 20000 bytes. The actual class name \
of the byte type
\t\t\t   in .NET is System.Byte, so a 1-dimensional array of these is called \
System.Byte[].
\t\t\t   NETNew always returns an object reference, so we get back the \
object, not a list of
\t\t\t   20000 zeros.
\t\t\t*)
\t\t\tmanagedArray = NETNew[\"System.Byte[]\", 20000];
\t\t\tresult = \"\";
\t\t\tWhile[(cnt = bzread[bzPtr, managedArray, 20000]) > 0,
\t\t\t\t(* NETObjectToExpression takes the array object and forces it to be \
returned to
\t\t\t\t   Mathematica as a list of numbers instead of an object reference. \
We encoded
\t\t\t\t   the data using the UTF8 format in the Compress function, so we \
must use that
\t\t\t\t   format when we decode from an array of bytes back into a \
Mathematica string.
\t\t\t\t*)
\t\t\t\tresult = result <> \
FromCharacterCode[Take[NETObjectToExpression[managedArray], cnt], \"UTF8\"]
\t\t\t];
\t\t\tIf[cnt == -1,
\t\t\t\t(* In this call, errCode is an \"out int\" parameter. It doesn't \
matter what value it
\t\t\t\t   has going in, and it will be assigned an integer value during the \
call.
\t\t\t\t*)
\t\t\t\terrString = bzerror[bzPtr, errCode];
\t\t\t\tMessage[Uncompress::bzerr, errCode, errString];
\t\t\t\tresult = $Failed
\t\t\t];
\t\t\tbzclose[bzPtr];
\t\t\tresult
\t\t]
\t]\
\>", "Input",
 InitializationCell->True],

Cell[TextData[{
 "Here is the second tehnique for dealing with the block of memory argument \
that ",
 StyleBox["BZ2_bzread", "Output"],
 " requires. This time we make this argument an ",
 StyleBox["IntPtr", "Output"],
 " that holds the address of a block of memory. We must manually allocate and \
later deallocate this block of memory. This technique is more complicated, \
but it is presented because it demonstrates some useful techniques for \
dealing directly with memory and pointers in ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " code. Several useful methods are found in the ",
 StyleBox["System.Runtime.InteropServices.Marshal", "Output"],
 " class. We use the ",
 StyleBox["Marshal.AllocHGlobal() ", "Output"],
 "function to allocate a block of memory and ",
 StyleBox["Marshal.FreeHGlobal()", "Output"],
 " to free it. These names will be familiar to Windows programmers as simple \
wrappers for the Windows API functions ",
 StyleBox["GlobalAlloc", "Output"],
 " and ",
 StyleBox["GlobalFree", "Output"],
 ". ",
 StyleBox["Marshal.AllocHGlobal()", "Output"],
 " returns an ",
 StyleBox["IntPtr", "Output"],
 " that points to the allocated memory. This ",
 StyleBox["IntPtr", "Output"],
 " is passed to ",
 StyleBox["bzread", "Input"],
 " which writes into it. The final step is to copy the data in that raw \
memory block into a managed array (that is, a real .NET array) that can be \
returned to ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " as a list."
}], "Text"],

Cell["\<\
(* Here is a new definition for calling the BZ2_bzread DLL function. The \
second argument is different
   from the one used in the bzread definition.
*)
bzread2 = DefineDLLFunction[\"BZ2_bzread\", fullDllPath, \"int\", \
{\"IntPtr\", \"IntPtr\", \"int\"}];

Uncompress2[file_String] :=
\tNETBlock[
\t\tModule[{bzPtr, bufPtr, cnt, result, managedArray, errString, errCode},
\t\t\tLoadNETType[\"System.Runtime.InteropServices.Marshal\"];
\t\t\tbzPtr = bzopen[file, \"r\"];
\t\t\t(* Test if the IntPtr was NULL. *)
\t\t\tIf[bzPtr@ToInt32[] == 0,
\t\t\t\tMessage[Uncompress::fopen, file];
\t\t\t\tReturn[$Failed]
\t\t\t];
\t\t\t(* Allocate a block of memory 20000 bytes long and get an IntPtr to it. \
This IntPtr is what
\t\t\t   we will pass in to bzread2.
\t\t\t*)
\t\t\tbufPtr = Marshal`AllocHGlobal[20000];
\t\t\t(* To return the data to Mathematica, we need to copy the raw byte data \
from memory into
\t\t\t   a managed array. This array is used for that purpose.
\t\t\t*)
\t\t\tmanagedArray = NETNew[\"System.Byte[]\", 20000];
\t\t\tresult = \"\";
\t\t\tWhile[(cnt = bzread2[bzPtr, bufPtr, 20000]) > 0,
\t\t\t\t(* Here we copy the data from the block of memory into the managed \
array. *)
\t\t\t\tMarshal`Copy[bufPtr, managedArray, 0, cnt];
\t\t\t\t(* See the notes for this line in the original Uncompress function \
above. *)
\t\t\t\tresult = result <> \
FromCharacterCode[Take[NETObjectToExpression[managedArray], cnt], \"UTF8\"]
\t\t\t];
\t\t\t(* Deallocate the block we allocated earlier. *)
\t\t\tMarshal`FreeHGlobal[bufPtr];
\t\t\tIf[cnt == -1,
\t\t\t\terrString = bzerror[bzPtr, errCode];
\t\t\t\tMessage[Uncompress::bzerr, errCode, errString];
\t\t\t\tresult = $Failed
\t\t\t];
\t\t\tbzclose[bzPtr];
\t\t\tresult
\t\t]
\t]\
\>", "Input",
 InitializationCell->True]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell["Example", "Section"],

Cell[TextData[{
 "Let's compress and uncompress a file from the ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " distribution."
}], "Text"],

Cell["\<\
strm = OpenRead[ToFileName[{$TopDirectory, \"Documentation\", \"English\", \
\"MainBook\"}, \"Contents.nb\"], DOSTextFormat->False];
origFileData = Import[strm, \"Text\"];
Close[strm];\
\>", "Input"],

Cell[CellGroupData[{

Cell["StringLength[origFileData]", "Input"],

Cell[BoxData["109582"], "Output"]
}, Open  ]],

Cell[TextData[{
 "Now we compress that data to a temporary file. Calling ",
 StyleBox["Close[OpenTemporary[]]", "Input"],
 " is just a trick for generating an unused filename in the temporary \
directory."
}], "Text"],

Cell[CellGroupData[{

Cell["\<\
tempFileName = Close[OpenTemporary[]];
Compress[tempFileName, origFileData]\
\>", "Input"],

Cell[BoxData["109582"], "Output"]
}, Open  ]],

Cell["Check the size of the compressed version:", "Text"],

Cell[CellGroupData[{

Cell["ByteCount /. FileInformation[tempFileName]", "Input"],

Cell[BoxData["10711"], "Output"]
}, Open  ]],

Cell["\<\
That's about 10:1 compression! Now check that we can uncompress and get the \
original data back:\
\>", "Text"],

Cell[CellGroupData[{

Cell["Uncompress[tempFileName] == origFileData", "Input"],

Cell[BoxData["True"], "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
},
AutoGeneratedPackage->None,
WindowSize->{987, 664},
WindowMargins->{{1, Automatic}, {Automatic, 0}},
DockedCells->(None& ),
FrontEndVersion->"6.0 for Mac OS X PowerPC (32-bit) (February 21, 2007)",
StyleDefinitions->"Default.nb"
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[CellGroupData[{
Cell[590, 23, 2538, 77, 296, "Title"],
Cell[CellGroupData[{
Cell[3153, 104, 38, 0, 67, "Section"],
Cell[3194, 106, 64, 1, 30, "Input",
 InitializationCell->True],
Cell[3261, 109, 648, 15, 71, "Text"],
Cell[3912, 126, 162, 4, 30, "Input",
 InitializationCell->True],
Cell[4077, 132, 370, 10, 53, "Text"],
Cell[CellGroupData[{
Cell[4472, 146, 28, 0, 38, "Subsection"],
Cell[4503, 148, 122, 4, 33, "Text"],
Cell[4628, 154, 79, 0, 42, "Program"],
Cell[4710, 156, 553, 14, 72, "Text"],
Cell[5266, 172, 158, 4, 30, "Input",
 InitializationCell->True],
Cell[5427, 178, 106, 3, 33, "Text"],
Cell[5536, 183, 455, 11, 120, "Input",
 Evaluatable->False]
}, Open  ]],
Cell[CellGroupData[{
Cell[6028, 199, 29, 0, 38, "Subsection"],
Cell[6060, 201, 123, 4, 33, "Text"],
Cell[6186, 207, 69, 0, 42, "Program"],
Cell[6258, 209, 1047, 26, 90, "Text"],
Cell[7308, 237, 156, 4, 30, "Input",
 InitializationCell->True]
}, Open  ]],
Cell[CellGroupData[{
Cell[7501, 246, 28, 0, 38, "Subsection"],
Cell[7532, 248, 122, 4, 33, "Text"],
Cell[7657, 254, 69, 0, 42, "Program"],
Cell[7729, 256, 528, 15, 53, "Text"],
Cell[8260, 273, 154, 4, 30, "Input",
 InitializationCell->True]
}, Open  ]],
Cell[CellGroupData[{
Cell[8451, 282, 29, 0, 38, "Subsection"],
Cell[8483, 284, 123, 4, 33, "Text"],
Cell[8609, 290, 50, 0, 42, "Program"],
Cell[8662, 292, 136, 4, 30, "Input",
 InitializationCell->True]
}, Open  ]],
Cell[CellGroupData[{
Cell[8835, 301, 29, 0, 38, "Subsection"],
Cell[8867, 303, 123, 4, 33, "Text"],
Cell[8993, 309, 65, 0, 42, "Program"],
Cell[9061, 311, 2006, 46, 185, "Text"],
Cell[11070, 359, 150, 4, 30, "Input",
 InitializationCell->True]
}, Open  ]],
Cell[CellGroupData[{
Cell[11257, 368, 43, 0, 38, "Subsection"],
Cell[11303, 370, 234, 5, 48, "Input",
 InitializationCell->True],
Cell[11540, 377, 684, 15, 91, "Text"],
Cell[12227, 394, 783, 25, 390, "Input",
 InitializationCell->True]
}, Open  ]],
Cell[CellGroupData[{
Cell[13047, 424, 45, 0, 38, "Subsection"],
Cell[13095, 426, 238, 5, 48, "Input",
 InitializationCell->True],
Cell[13336, 433, 1251, 26, 169, "Text"],
Cell[14590, 461, 1634, 47, 642, "Input",
 InitializationCell->True],
Cell[16227, 510, 1500, 37, 148, "Text"],
Cell[17730, 549, 1790, 50, 750, "Input",
 InitializationCell->True]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[19569, 605, 26, 0, 37, "Section"],
Cell[19598, 607, 145, 5, 26, "Text"],
Cell[19746, 614, 209, 5, 50, "Input"],
Cell[CellGroupData[{
Cell[19980, 623, 43, 0, 24, "Input"],
Cell[20026, 625, 33, 0, 24, "Output"]
}, Open  ]],
Cell[20074, 628, 217, 5, 26, "Text"],
Cell[CellGroupData[{
Cell[20316, 637, 100, 3, 37, "Input"],
Cell[20419, 642, 33, 0, 24, "Output"]
}, Open  ]],
Cell[20467, 645, 57, 0, 26, "Text"],
Cell[CellGroupData[{
Cell[20549, 649, 59, 0, 24, "Input"],
Cell[20611, 651, 32, 0, 24, "Output"]
}, Open  ]],
Cell[20658, 654, 121, 3, 26, "Text"],
Cell[CellGroupData[{
Cell[20804, 661, 57, 0, 24, "Input"],
Cell[20864, 663, 31, 0, 24, "Output"]
}, Open  ]]
}, Open  ]]
}, Open  ]]
}
]
*)

(* End of internal cache information *)

